#ifndef __IControl__
#define __IControl__

#include <IO/CXMLNode.hpp>
#include <Basics/CCountedObject.hpp>
#include <MathTools/CDimension.hpp>
#include <MathTools/CRect.hpp>
#include "../Graphics/IImage.hpp"
#include "../Graphics/CGraphics.hpp"
#include "../Listeners/IActionListener.hpp"
#include "../Listeners/IMouseListener.hpp"
#include "../Listeners/IKeyboardListener.hpp"
#include "../Listeners/IDropFileListener.hpp"
#include "IControlRoot.hpp"

//	===========================================================================

using Exponent::IO::CXMLNode;
using Exponent::Basics::CCountedObject;
using Exponent::MathTools::CDimension;
using Exponent::MathTools::CRect;
using Exponent::GUI::Graphics::IImage;
using Exponent::GUI::Graphics::CGraphics;
using Exponent::GUI::Listeners::IActionListener;
using Exponent::GUI::Controls::IControlRoot;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @interface IControl IControl.hpp
			 * @brief Interface to a control
			 *
			 * Controls are leightweight classes that can draw themselves and can handle input from various sources (depending on what listener you register as)\n
			 * Each control has several key function and attributes:\n
			 * <UL>
			 *	<LI>Each control has a unique idnetifier. This must be unique within this controls root control. This is how you know which control is sending which event (unless the event includes a reference to the control as @see CActionEvent)
			 *	<LI>Each control must be seated inside a control root of some type. @see IControlRoot
			 *	<LI>Controls have a value. This value is a double precision floating point number which should be in the range of 0 - 1
			 *	<LI>Controls have an area. This area is relative to their parents top left, so an area of 0,0,100,100 would place a square control (100px in size) in the top left of its parent
			 *	<LI>Controls can force themselves to be redrawn using the update call
			 * </UL>
			 *
			 * @see CGraphics
			 * @see IControlRoot
			 * @see CControl
			 *
			 * @date 19/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: IControl.hpp,v 1.7 2007/02/08 21:07:54 paul Exp $
			 */
			interface IControl : public ICountedObject, public IMouseListener, public IKeyboardListener, public IDropFileListener
			{

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * Construction
				 */
				IControl() { }

				/**
				 * Destruction
				 */
				virtual ~IControl() { }

//	===========================================================================

				/**
				 * Set the root control that this control is sitting on
				 * @param controlRoot The root control
				 */
				virtual void setControlRoot(IControlRoot *controlRoot) = 0;

				/**
				 * Get the root control
				 * @retval IControlRoot* The control root
				 */
				virtual IControlRoot *getRootControl() const = 0;

//	===========================================================================

				/**
				 * Set the parent control
				 * @param parent The parent container control
				 */
				virtual void setParentControl(IControl *parent) = 0;

				/**
				 * Get the parent control
				 * @retval IControl* The parent control
				 */
				virtual IControl *getParentControl() const = 0;

//	===========================================================================

				/**
				 * Register the action listener
				 * @param listener The action listener for this control
				 */
				virtual void registerActionListener(IActionListener *listener) = 0;

//	===========================================================================

				/**
				 * Does the control accept mouse messages?
				 * @param mouseEnabled True if the control accepts mouse messages, false otherwise
				 */
				virtual void controlIsMouseEnabled(const bool mouseEnabled = true) = 0;

				/**
				 * Does the control accept dropped files?
				 * @param dropEnabled True if the control accepts dropped file messages, false otherwise
				 */
				virtual void controlIsDropFileEnabled(const bool dropEnabled = true) = 0;

				/**
				 * Does the control accept key messages
				 * @param keyEnabled True if the control accepts keyboard messages, false otherwise
				 */
				virtual void controlIsKeyEnabled(const bool keyEnabled = true) = 0;

				/**
				 * Reserved for Future use
				 * @param identifier The identifier of the control that this control is enabled for
				 */
				virtual void controlIsEnabledFor(const CString &identifier) = 0;

//	===========================================================================

				/**
				 * Is the control mouse enabled
				 * @retval bool True if control accepts mouse handling functionality, false otherwise
				 */
				virtual bool isMouseEnabled() const = 0;

				/**
				 * Is the control drop enabled
				 * @retval bool True if the control can accept dropped files, false otherwise
				 */
				virtual bool isDropFileEnabled() const = 0;

				/**
				 * Is the control key enabled
				 * @retval bool True if the control can accept keyboard handling messages, false otherwise
				 */
				virtual bool isKeyEnabled() const = 0;

				/**
				 * Is the control enabled for (future use)
				 * @param identifier The identifier for the usage required.
				 * @note Not currently enabled, for future use only
				 */
				virtual bool isEnabledFor(const CString &identifier) = 0;

				/**
				 * Does the control require a right click feed?
				 * @retval bool True if the control needs right click information, false otherwise
				 */
				virtual bool needsRightClick() const = 0;

//	===========================================================================

				/**
				 * Serialise the control from a filename
				 * @param node The node that you should setup
				 * @retval bool True if saved to node properly, false otherwise
				 * @note FOR FUTURE USE, DO NOT USE YET!
				 */
				virtual bool serializeToNode(CXMLNode &node) = 0;

				/**
				 * Serialise the control from a filename
				 * @param node The node that you should setup from
				 * @retval bool True if loaded from node properly, false otherwise
				 * @note FOR FUTURE USE, DO NOT USE YET!
				 */
				virtual bool serializeFromNode(CXMLNode &node) = 0;

//	===========================================================================

				/**
				 * Get the id of the control
				 * @retval long The unique control identifier
				 */
				virtual long getUniqueId() const = 0;

				/**
				 * Set the id of the control
				 * @param id The unique id of the control
				 */
				virtual void setUniqueId(const long id) = 0;

//	===========================================================================

				/**
				 * Set the vlaue of the control
				 * @param value The value of the control (0 - 1 ranged)
				 */
				virtual void setValue(const double value) = 0;

				/**
				 * Get the value of the control
				 * @retval double The value of the control (0 - 1 ranged)
				 */
				virtual double getValue() const = 0;

//	===========================================================================

				/**
				 * Set the area of the control
				 * @param area The are of the control relative to the root control
				 */
				virtual void setArea(const CRect &area) = 0;

				/**
				 * Get the position relative to their root
				 * @retval const CRect& The controls area relative to the root control
				 */
				virtual const CRect &getArea() const = 0;

				/**
				 * Get the position on the window
				 * @note The returned position may not be the complete area of the control, ie it may be clipped or otherwise\n
				 * altered. This function is mainly included for root control use and is *not* recommended for internal control\n
				 * use, except in exceptional circumstance.
				 * @see IControlRoot::getGlobalCoordinatesOfControl
				 * @see IControlRoot::getWindowCoordinatesOfControl
				 * @retval const CRect& The absolute rectangle of the control - ie the actual on screen position fully adjusted
				 */
				virtual const CRect &getAbsoluteRect() const = 0;

				/**
				 * Get the size of the control
				 * @retval const CDimension& The dimension of the control
				 */
				virtual const CDimension &getDimension() const = 0;

				/**
				 * Is this control a container of other controls
				 * @retval bool True if the control is a control container, false otherwise
				 */
				virtual bool isContainer() const = 0;

//	===========================================================================

				/**
				 * Update the control (redraw it)
				 */
				virtual void update() = 0;

				/**
				 * Draw the controls
				 * @param graphics The graphics context
				 * @see CControl::drawControl
				 */
				virtual void drawControl(CGraphics &graphics) = 0;

//	===========================================================================

				/**
				 * Enable a control
				 */
				virtual void enableControl()  = 0;

				/**
				 * Disable the control
				 */
				virtual void disableControl() = 0;

//	===========================================================================

				/**
				 * Set the primary (usually background) image
				 * @param theImage The primary image for the control
				 */
				virtual void setPrimaryImage(IImage *theImage) = 0;

				/**
				 * Set the disabled image
				 * @param theImage The disabled image for the control
				 */
				virtual void setDisabledImage(IImage *theImage) = 0;
			};
		}
	}
}
#endif	// End of IControl.hpp